﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using InfoWorld.HL7.ITS;
using BMS.Facade.Data;
using System.Web.Mvc;
using System.ComponentModel.DataAnnotations;
using BMS.Web.App_GlobalResource;
using BMS.Utils;
using System.Text;
using BMS.Web.Controllers.Shared;

namespace BMS.Web.Models
{
    /// <summary>
    /// Ems Notification View Model class.
    /// </summary>
    public class EmsNotificationViewModel
    {
        #region Public Members
        /// <summary>
        /// Gets or sets the logged user.
        /// </summary>
        /// <value>
        /// The logged user.
        /// </value>
        public User LoggedUser { get; set; }

        /// <summary>
        /// Gets or sets the ward group list.
        /// </summary>
        /// <value>
        /// The ward group list.
        /// </value>
        public IList<Division> WardGroupList { get; set; }

        /// <summary>
        /// Gets or sets the ward group object selected.
        /// </summary>
        /// <value>
        /// The ward group object selected.
        /// </value>
        public Division WardGroupSelected { get; set; }

        /// <summary>
        /// Gets or sets the location.
        /// </summary>
        /// <value>
        /// The location.
        /// </value>
        [ValidateXss]
        public string Location { get; set; }

        /// <summary>
        /// Gets or sets the location list.
        /// </summary>
        /// <value>
        /// The location list.
        /// </value>
        public IList<EMSNotification> EMSNotificationList
        {
            set
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    EMSNotificationListExtended = new List<EMSNotificationItemViewModel>();
                    string _notificationText = string.Empty;
                    // generate the extended ems notification list(with the notification text properties).
                    if (value != null)
                    {
                        foreach (EMSNotification item in value)
                        {
                            EMSNotificationListExtended.Add(new EMSNotificationItemViewModel()
                            {
                                Model = item,
                                EMSNotificationText = GetNotificationText(item.EMSCleaned, item.EMSDirty),
                                BedControllerNotificationText = GetNotificationText(item.BedControllerCleaned, item.BedControllerDirty),
                                OtherNotificationText = GetNotificationText(item.VistaGroupCleaned, item.VistaGroupDirty)
                            });
                        }
                    }
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }

        /// <summary>
        /// Gets or sets the EMS notification list extended(containing the notification text properties).
        /// </summary>
        /// <value>
        /// The EMS notification list extended.
        /// </value>
        public IList<EMSNotificationItemViewModel> EMSNotificationListExtended { get; set; }

        /// <summary>
        /// Gets or sets the name of the facility.
        /// </summary>
        /// <value>
        /// The name of the facility.
        /// </value>
        public static string FacilityName { get; set; }

        /// <summary>
        /// Gets or sets the flags list.
        /// </summary>
        /// <value>
        /// The flags list.
        /// </value>
        public CDList FlagsList { get; set; }

        /// <summary>
        /// Gets or sets the ems dirty.
        /// </summary>
        /// <value>
        /// The ems dirty.
        /// </value>
        public CD EmsDirty { get; set; }

        /// <summary>
        /// Gets or sets the ems clean.
        /// </summary>
        /// <value>
        /// The ems clean.
        /// </value>
        public CD EmsClean { get; set; }

        /// <summary>
        /// Gets or sets the bed controller dirty.
        /// </summary>
        /// <value>
        /// The bed controller dirty.
        /// </value>
        public CD BedControllerDirty { get; set; }

        /// <summary>
        /// Gets or sets the bed controller clean.
        /// </summary>
        /// <value>
        /// The bed controller clean.
        /// </value>
        public CD BedControllerClean { get; set; }

        /// <summary>
        /// Gets or sets the vista dirty.
        /// </summary>
        /// <value>
        /// The vista dirty.
        /// </value>
        public CD VistaDirty { get; set; }

        /// <summary>
        /// Gets or sets the vista clean.
        /// </summary>
        /// <value>
        /// The vista clean.
        /// </value>
        public CD VistaClean { get; set; }

        /// <summary>
        /// Gets or sets the ems email.
        /// </summary>
        /// <value>
        /// The ems email.
        /// </value>
        [RequiredIf("EmsClean.code", "Yes", ErrorMessage = "No email addresses are entered. Please enter the Valid Email address")]
        [RequiredIf("EmsDirty.code", "Yes", ErrorMessage = "No email addresses are entered. Please enter the Valid Email address")]
        [StringLength(150, ErrorMessage = "Addresses string too long. The total number of characters allowed is 150 per text box with no spaces...")]
        [RegularExpression(Constants.EMAIL_GROUP_REGULAR_EXPRESSION, ErrorMessage = "Invalid email.")]
        public string EmsEmail { get; set; }

        /// <summary>
        /// Gets or sets the bed controller email.
        /// </summary>
        /// <value>
        /// The bed controller email.
        /// </value>
        [RequiredIf("BedControllerClean.code", "Yes", ErrorMessage = "No email addresses are entered. Please enter the Valid Email address")]
        [RequiredIf("BedControllerDirty.code", "Yes", ErrorMessage = "No email addresses are entered. Please enter the Valid Email address")]
        [StringLength(150, ErrorMessage = "Addresses string too long. The total number of characters allowed is 150 per text box with no spaces...")]
        [RegularExpression(Constants.EMAIL_GROUP_REGULAR_EXPRESSION, ErrorMessage = "Invalid email.")]
        public string BedControllerEmail { get; set; }

        /// <summary>
        /// Gets or sets the vista email.
        /// </summary>
        /// <value>
        /// The vista email.
        /// </value>
        [RequiredIf("VistaClean.code", "Yes", ErrorMessage = "No email addresses are entered. Please enter the Valid Email address")]
        [RequiredIf("VistaDirty.code", "Yes", ErrorMessage = "No email addresses are entered. Please enter the Valid Email address")]
        [StringLength(150, ErrorMessage = "Addresses string too long. The total number of characters allowed is 150 per text box with no spaces...")]
        [RegularExpression(Constants.EMAIL_GROUP_REGULAR_EXPRESSION, ErrorMessage = "Invalid email.")]
        public string VistaEmail { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether this instance is add operation.
        /// </summary>
        /// <value>
        /// 	<c>true</c> if this instance is add operation; otherwise, <c>false</c>.
        /// </value>
        public bool IsAddOperation { get; set; }

        /// <summary>
        /// Gets or sets the EMS already exists display.
        /// </summary>
        /// <value>
        /// The EMS already exists display.
        /// </value>
        public string EMSAlreadyExistsDisplay { get; set; }

        /// <summary>
        /// Gets or sets the EMS add edit display.
        /// </summary>
        /// <value>
        /// The EMS add edit display.
        /// </value>
        public string EMSAddEditDisplay { get; set; }

        /// <summary>
        /// Gets or sets the display message when successfully add edit notification.
        /// </summary>
        /// <value>
        /// The display message when successfully add edit notification.
        /// </value>
        public string DisplayMessageWhenSuccessfullyAddEditNotification { get; set; }

        /// <summary>
        /// Gets or sets the display message when ems notification already exist.
        /// </summary>
        /// <value>
        /// The display message when ems notification already exist.
        /// </value>
        public string DisplayMessageWhenEmsNotificationAlreadyExist { get; set; }

        /// <summary>
        /// Gets or sets the header.
        /// </summary>
        /// <value>
        /// The header.
        /// </value>
        public string Header { get; set; }

        /// <summary>
        /// Gets or sets the display text for submit button.
        /// </summary>
        /// <value>
        /// The display text for submit button.
        /// </value>
        public string DisplayTextForSubmitButton { get; set; }
        #endregion

        #region Methods
        /// <summary>
        /// Translates from view model object to domain object.
        /// </summary>
        /// <param name="emsNotificationViewModel">The ems notification view model.</param>
        /// <returns>Location object.</returns>
        public static EMSNotification TranslateFromViewModelToDomainObject(EmsNotificationViewModel emsNotificationViewModel)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                AntiXssEncoder xss = new AntiXssEncoder();
                EMSNotification emsNotification = new EMSNotification();
                emsNotification.Location = xss.Decode(emsNotificationViewModel.Location);
                emsNotification.EMSEmail = emsNotificationViewModel.EmsEmail;
                emsNotification.BedControllerEmail = emsNotificationViewModel.BedControllerEmail;
                emsNotification.VistaGroupMail = emsNotificationViewModel.VistaEmail;
                emsNotification.EMSCleaned = emsNotificationViewModel.EmsClean;
                emsNotification.EMSDirty = emsNotificationViewModel.EmsDirty;
                emsNotification.BedControllerCleaned = emsNotificationViewModel.BedControllerClean;
                emsNotification.BedControllerDirty = emsNotificationViewModel.BedControllerDirty;
                emsNotification.VistaGroupCleaned = emsNotificationViewModel.VistaClean;
                emsNotification.VistaGroupDirty = emsNotificationViewModel.VistaDirty;

                return emsNotification;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public string GetNotificationText(CD cleaned, CD dirty)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                StringBuilder result = new StringBuilder();
                if (cleaned.displayName.Equals(Constants.Yes) && dirty.displayName.Equals(Constants.Yes))
                {
                    result.Append(Strings.Dirty);
                    result.Append("/");
                    result.Append(Strings.Clean);
                    return result.ToString();
                }
                if (cleaned.displayName.Equals(Constants.Yes))
                    return Strings.Clean;
                if (dirty.displayName.Equals(Constants.Yes))
                    return Strings.Dirty;

                return Strings.None;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

    }

    /// <summary>
    /// Additional view model class containing ems notification object and the notification text.
    /// </summary>
    public class EMSNotificationItemViewModel : BaseViewModel
    {
        /// <summary>
        /// Gets or sets the ems notification object.
        /// </summary>
        /// <value>
        /// The ems notification object.
        /// </value>
        public EMSNotification Model { get; set; }

        /// <summary>
        /// Gets or sets the location extension.
        /// </summary>
        /// <value>
        /// The location extension.
        /// </value>
        public string LocationExtension { get; set; }

        /// <summary>
        /// Gets or sets the ems notification text.
        /// </summary>
        /// <value>
        /// The ems notification text.
        /// </value>
        public string EMSNotificationText { get; set; }

        /// <summary>
        /// Gets or sets the bed controller notification text.
        /// </summary>
        /// <value>
        /// The bed controller notification text.
        /// </value>
        public string BedControllerNotificationText { get; set; }

        /// <summary>
        /// Gets or sets the other notification text.
        /// </summary>
        /// <value>
        /// The other notification text.
        /// </value>
        public string OtherNotificationText { get; set; }
    }
}